@extends('components.app-layout')

@section('title', 'Create Category')
@section('subtitle', 'Add a new inventory category')

@section('content')
<div class="min-h-screen bg-gradient-to-br from-slate-50 via-emerald-50 to-green-50 py-8">
    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 space-y-8">
        
        <!-- Header Section -->
        <div class="relative overflow-hidden bg-gradient-to-r from-emerald-600 via-green-600 to-teal-600 rounded-3xl shadow-2xl">
            <div class="absolute inset-0 bg-black opacity-5"></div>
            <div class="absolute top-0 right-0 w-64 h-64 bg-white opacity-5 rounded-full transform translate-x-32 -translate-y-32"></div>
            <div class="absolute bottom-0 left-0 w-48 h-48 bg-white opacity-5 rounded-full transform -translate-x-24 translate-y-24"></div>
            
            <div class="relative px-8 py-12">
                <div class="max-w-4xl mx-auto">
                    <div class="flex items-center space-x-6">
                        <div class="w-16 h-16 bg-white bg-opacity-20 backdrop-blur-sm rounded-2xl flex items-center justify-center shadow-xl">
                            <i class="fas fa-plus text-2xl text-white"></i>
                        </div>
                        <div>
                            <h1 class="text-4xl font-bold text-white mb-2">Create Category</h1>
                            <p class="text-xl text-white opacity-90">Add a new category to organize your inventory items</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Create Form -->
        <div class="bg-white/90 backdrop-blur-sm rounded-2xl shadow-xl border border-gray-100/50 p-8">
            <form action="{{ route('inventory.categories.store') }}" method="POST" class="space-y-8">
                @csrf

                <!-- Basic Information Section -->
                <div>
                    <div class="flex items-center space-x-3 mb-6">
                        <div class="w-10 h-10 bg-gradient-to-r from-emerald-600 to-green-600 rounded-xl flex items-center justify-center">
                            <i class="fas fa-info-circle text-white"></i>
                        </div>
                        <h2 class="text-2xl font-bold text-gray-900">Basic Information</h2>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Category Name -->
                        <div>
                            <label for="name" class="block text-sm font-medium text-gray-700 mb-2">Category Name *</label>
                            <input type="text" name="name" id="name" value="{{ old('name') }}" 
                                   class="block w-full px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-emerald-500/20 focus:border-emerald-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md @error('name') border-red-500 @enderror" 
                                   placeholder="Enter category name"
                                   required>
                            @error('name')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Status -->
                        <div>
                            <label for="is_active" class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <div class="relative">
                                <select name="is_active" id="is_active" 
                                        class="block w-full px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-emerald-500/20 focus:border-emerald-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md appearance-none @error('is_active') border-red-500 @enderror">
                                    <option value="1" {{ old('is_active', true) ? 'selected' : '' }}>Active</option>
                                    <option value="0" {{ !old('is_active', true) ? 'selected' : '' }}>Inactive</option>
                                </select>
                                <div class="absolute inset-y-0 right-0 flex items-center pr-4 pointer-events-none">
                                    <i class="fas fa-chevron-down text-gray-400"></i>
                                </div>
                            </div>
                            @error('is_active')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Description -->
                        <div class="md:col-span-2">
                            <label for="description" class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                            <textarea name="description" id="description" rows="3" 
                                      class="block w-full px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-emerald-500/20 focus:border-emerald-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md @error('description') border-red-500 @enderror" 
                                      placeholder="Describe this category...">{{ old('description') }}</textarea>
                            @error('description')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Visual Settings Section -->
                <div>
                    <div class="flex items-center space-x-3 mb-6">
                        <div class="w-10 h-10 bg-gradient-to-r from-purple-600 to-pink-600 rounded-xl flex items-center justify-center">
                            <i class="fas fa-palette text-white"></i>
                        </div>
                        <h2 class="text-2xl font-bold text-gray-900">Visual Settings</h2>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Category Color -->
                        <div>
                            <label for="color" class="block text-sm font-medium text-gray-700 mb-2">Category Color</label>
                            <div class="flex items-center space-x-3">
                                <input type="color" name="color" id="color" value="{{ old('color', '#3B82F6') }}" 
                                       class="w-16 h-12 rounded-xl border-2 border-gray-200 cursor-pointer @error('color') border-red-500 @enderror"
                                       onchange="updateColorPreview()">
                                <input type="text" id="color_text" value="{{ old('color', '#3B82F6') }}" 
                                       class="flex-1 px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-purple-500/20 focus:border-purple-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md font-mono"
                                       onchange="updateColorPicker()"
                                       placeholder="#3B82F6">
                            </div>
                            @error('color')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                            <p class="mt-1 text-xs text-gray-500">Choose a color that represents this category</p>
                        </div>

                        <!-- Category Icon -->
                        <div>
                            <label for="icon" class="block text-sm font-medium text-gray-700 mb-2">Category Icon</label>
                            <div class="flex items-center space-x-3">
                                <div id="icon_preview" class="w-12 h-12 rounded-xl flex items-center justify-center shadow-lg" style="background: linear-gradient(135deg, #3B82F6, #3B82F699)">
                                    <i class="{{ old('icon', 'fas fa-tag') }} text-white text-lg"></i>
                                </div>
                                <input type="text" name="icon" id="icon" value="{{ old('icon', 'fas fa-tag') }}" 
                                       class="flex-1 px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-purple-500/20 focus:border-purple-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md font-mono @error('icon') border-red-500 @enderror"
                                       onchange="updateIconPreview()"
                                       placeholder="fas fa-tag">
                            </div>
                            @error('icon')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                            <p class="mt-1 text-xs text-gray-500">Use FontAwesome icon classes (e.g., fas fa-tag, fas fa-boxes)</p>
                        </div>
                    </div>

                    <!-- Icon Suggestions -->
                    <div class="mt-6">
                        <label class="block text-sm font-medium text-gray-700 mb-3">Popular Icons</label>
                        <div class="grid grid-cols-6 md:grid-cols-12 gap-3">
                            @php
                                $popularIcons = [
                                    'fas fa-tag', 'fas fa-boxes', 'fas fa-tools', 'fas fa-utensils',
                                    'fas fa-book', 'fas fa-microphone', 'fas fa-chair', 'fas fa-spray-can',
                                    'fas fa-laptop', 'fas fa-music', 'fas fa-camera', 'fas fa-gamepad',
                                    'fas fa-car', 'fas fa-home', 'fas fa-heart', 'fas fa-star',
                                    'fas fa-gift', 'fas fa-shopping-cart', 'fas fa-coffee', 'fas fa-leaf',
                                    'fas fa-lightbulb', 'fas fa-paint-brush', 'fas fa-wrench', 'fas fa-shield-alt'
                                ];
                            @endphp
                            @foreach($popularIcons as $iconClass)
                                <button type="button" onclick="selectIcon('{{ $iconClass }}')" 
                                        class="w-12 h-12 bg-gray-100 hover:bg-gray-200 rounded-xl flex items-center justify-center transition-colors group">
                                    <i class="{{ $iconClass }} text-gray-600 group-hover:text-gray-800"></i>
                                </button>
                            @endforeach
                        </div>
                    </div>

                    <!-- Color Presets -->
                    <div class="mt-6">
                        <label class="block text-sm font-medium text-gray-700 mb-3">Color Presets</label>
                        <div class="grid grid-cols-8 md:grid-cols-16 gap-3">
                            @php
                                $colorPresets = [
                                    '#3B82F6', '#EF4444', '#10B981', '#F59E0B', '#8B5CF6', '#06B6D4',
                                    '#EC4899', '#84CC16', '#F97316', '#6366F1', '#14B8A6', '#F43F5E',
                                    '#22C55E', '#A855F7', '#0EA5E9', '#EAB308', '#DC2626', '#059669',
                                    '#7C3AED', '#0891B2', '#BE185D', '#65A30D', '#EA580C', '#4F46E5'
                                ];
                            @endphp
                            @foreach($colorPresets as $colorPreset)
                                <button type="button" onclick="selectColor('{{ $colorPreset }}')" 
                                        class="w-8 h-8 rounded-full border-2 border-gray-300 hover:border-gray-400 transition-colors"
                                        style="background-color: {{ $colorPreset }}">
                                </button>
                            @endforeach
                        </div>
                    </div>

                    <!-- Live Preview -->
                    <div class="mt-6 p-4 rounded-xl border-2 border-gray-200" id="category_preview" style="background: linear-gradient(135deg, #3B82F6, #3B82F699, #3B82F666)">
                        <div class="flex items-center space-x-3">
                            <div class="w-12 h-12 bg-white bg-opacity-20 backdrop-blur-sm rounded-xl flex items-center justify-center shadow-lg">
                                <i id="preview_icon" class="fas fa-tag text-white text-lg"></i>
                            </div>
                            <div>
                                <h3 class="text-white font-bold text-lg" id="preview_name">New Category</h3>
                                <p class="text-white/80 text-sm">Category Preview</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Category Examples -->
                <div>
                    <div class="flex items-center space-x-3 mb-6">
                        <div class="w-10 h-10 bg-gradient-to-r from-blue-600 to-indigo-600 rounded-xl flex items-center justify-center">
                            <i class="fas fa-lightbulb text-white"></i>
                        </div>
                        <h2 class="text-2xl font-bold text-gray-900">Category Ideas</h2>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        @php
                            $categoryExamples = [
                                ['name' => 'Office Supplies', 'icon' => 'fas fa-pencil-alt', 'color' => '#3B82F6', 'desc' => 'Stationery, paper, pens'],
                                ['name' => 'Audio/Visual', 'icon' => 'fas fa-microphone', 'color' => '#EF4444', 'desc' => 'Sound equipment, projectors'],
                                ['name' => 'Furniture', 'icon' => 'fas fa-chair', 'color' => '#10B981', 'desc' => 'Chairs, tables, storage'],
                                ['name' => 'Cleaning Supplies', 'icon' => 'fas fa-spray-can', 'color' => '#F59E0B', 'desc' => 'Detergents, tools'],
                                ['name' => 'Kitchen Items', 'icon' => 'fas fa-utensils', 'color' => '#8B5CF6', 'desc' => 'Cookware, utensils'],
                                ['name' => 'Books & Materials', 'icon' => 'fas fa-book', 'color' => '#06B6D4', 'desc' => 'Bibles, educational materials']
                            ];
                        @endphp
                        @foreach($categoryExamples as $example)
                            <button type="button" onclick="useExample('{{ $example['name'] }}', '{{ $example['icon'] }}', '{{ $example['color'] }}', '{{ $example['desc'] }}')"
                                    class="text-left p-4 bg-gradient-to-r from-gray-50 to-gray-100 rounded-xl border border-gray-200 hover:shadow-lg transition-all duration-200 group">
                                <div class="flex items-center space-x-3 mb-2">
                                    <div class="w-8 h-8 rounded-lg flex items-center justify-center" style="background: {{ $example['color'] }}">
                                        <i class="{{ $example['icon'] }} text-white text-sm"></i>
                                    </div>
                                    <h4 class="font-medium text-gray-900 group-hover:text-gray-700">{{ $example['name'] }}</h4>
                                </div>
                                <p class="text-sm text-gray-600">{{ $example['desc'] }}</p>
                            </button>
                        @endforeach
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="flex flex-col sm:flex-row justify-between items-center gap-4 pt-8 border-t border-gray-200">
                    <div class="flex space-x-4">
                        <a href="{{ route('inventory.categories.index') }}" class="inline-flex items-center px-6 py-3 bg-gray-100 text-gray-700 font-medium rounded-xl border border-gray-200 hover:bg-gray-200 transition-all duration-200 shadow-md hover:shadow-lg">
                            <i class="fas fa-arrow-left mr-2"></i>
                            Back to Categories
                        </a>
                        <a href="{{ route('inventory.index') }}" class="inline-flex items-center px-6 py-3 bg-gray-100 text-gray-700 font-medium rounded-xl border border-gray-200 hover:bg-gray-200 transition-all duration-200 shadow-md hover:shadow-lg">
                            <i class="fas fa-tachometer-alt mr-2"></i>
                            Dashboard
                        </a>
                    </div>
                    
                    <div class="flex space-x-4">
                        <button type="reset" onclick="resetForm()" class="inline-flex items-center px-6 py-3 bg-gray-600 text-white font-medium rounded-xl hover:bg-gray-700 transition-colors shadow-lg hover:shadow-xl">
                            <i class="fas fa-undo mr-2"></i>
                            Reset Form
                        </button>
                        <button type="submit" class="inline-flex items-center px-8 py-3 bg-gradient-to-r from-emerald-600 via-green-600 to-teal-600 text-white font-bold rounded-xl hover:from-emerald-700 hover:via-green-700 hover:to-teal-700 transition-all duration-300 shadow-xl hover:shadow-2xl transform hover:scale-105">
                            <i class="fas fa-plus mr-2"></i>
                            Create Category
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function updateColorPreview() {
    const colorInput = document.getElementById('color');
    const colorText = document.getElementById('color_text');
    const preview = document.getElementById('category_preview');
    const iconPreview = document.getElementById('icon_preview');
    
    const color = colorInput.value;
    colorText.value = color;
    
    preview.style.background = `linear-gradient(135deg, ${color}, ${color}99, ${color}66)`;
    iconPreview.style.background = `linear-gradient(135deg, ${color}, ${color}99)`;
}

function updateColorPicker() {
    const colorText = document.getElementById('color_text');
    const colorInput = document.getElementById('color');
    const preview = document.getElementById('category_preview');
    const iconPreview = document.getElementById('icon_preview');
    
    const color = colorText.value;
    if (/^#[0-9A-F]{6}$/i.test(color)) {
        colorInput.value = color;
        preview.style.background = `linear-gradient(135deg, ${color}, ${color}99, ${color}66)`;
        iconPreview.style.background = `linear-gradient(135deg, ${color}, ${color}99)`;
    }
}

function updateIconPreview() {
    const iconInput = document.getElementById('icon');
    const iconPreview = document.getElementById('icon_preview').querySelector('i');
    const previewIcon = document.getElementById('preview_icon');
    
    const iconClass = iconInput.value;
    iconPreview.className = iconClass + ' text-white text-lg';
    previewIcon.className = iconClass + ' text-white text-lg';
}

function selectIcon(iconClass) {
    const iconInput = document.getElementById('icon');
    iconInput.value = iconClass;
    updateIconPreview();
}

function selectColor(color) {
    const colorInput = document.getElementById('color');
    const colorText = document.getElementById('color_text');
    
    colorInput.value = color;
    colorText.value = color;
    updateColorPreview();
}

function useExample(name, icon, color, description) {
    document.getElementById('name').value = name;
    document.getElementById('icon').value = icon;
    document.getElementById('color').value = color;
    document.getElementById('color_text').value = color;
    document.getElementById('description').value = description;
    
    updateColorPreview();
    updateIconPreview();
    document.getElementById('preview_name').textContent = name;
}

function resetForm() {
    document.getElementById('name').value = '';
    document.getElementById('icon').value = 'fas fa-tag';
    document.getElementById('color').value = '#3B82F6';
    document.getElementById('color_text').value = '#3B82F6';
    document.getElementById('description').value = '';
    document.getElementById('is_active').value = '1';
    
    updateColorPreview();
    updateIconPreview();
    document.getElementById('preview_name').textContent = 'New Category';
}

// Update preview name when typing
document.getElementById('name').addEventListener('input', function() {
    document.getElementById('preview_name').textContent = this.value || 'New Category';
});

// Initialize previews
document.addEventListener('DOMContentLoaded', function() {
    updateColorPreview();
    updateIconPreview();
});
</script>
@endsection
