#!/usr/bin/env python3
"""
Generate Excel file with 1000 Ghanaian names for bulk member import
"""

import random
from datetime import datetime, timedelta
from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill, Alignment

# Ghanaian names (Akan day names and common names)
MALE_FIRST_NAMES = [
    'Kwame', 'Kwasi', 'Kwadwo', 'Kwabena', 'Yaw', 'Kofi', 'Kwaku',
    'Kojo', 'Kweku', 'Kofi', 'Kwame', 'Kwasi', 'Kwadwo', 'Kwabena',
    'Samuel', 'Emmanuel', 'Joseph', 'Daniel', 'David', 'Benjamin',
    'Isaac', 'Jacob', 'Michael', 'Stephen', 'Peter', 'Paul',
    'Francis', 'Eric', 'Prince', 'Ernest', 'Anthony', 'Augustine',
    'Felix', 'Albert', 'Alfred', 'Bernard', 'Charles', 'Dennis',
    'Edward', 'Frederick', 'George', 'Henry', 'James', 'John'
]

FEMALE_FIRST_NAMES = [
    'Akosua', 'Afua', 'Abena', 'Abenaa', 'Yaa', 'Efua', 'Ama',
    'Adjoa', 'Akua', 'Adwoa', 'Adwoa', 'Akosua', 'Afua', 'Abena',
    'Mary', 'Grace', 'Esther', 'Ruth', 'Sarah', 'Hannah',
    'Rebecca', 'Rachel', 'Elizabeth', 'Martha', 'Lydia', 'Deborah',
    'Joyce', 'Patience', 'Mercy', 'Faith', 'Hope', 'Charity',
    'Agnes', 'Alice', 'Betty', 'Comfort', 'Dorothy', 'Eunice',
    'Felicia', 'Gloria', 'Helen', 'Janet', 'Jennifer', 'Linda'
]

MIDDLE_NAMES = [
    'Mensah', 'Owusu', 'Boateng', 'Agyei', 'Agyemang', 'Amoah',
    'Adjei', 'Appiah', 'Osei', 'Gyasi', 'Frimpong', 'Acheampong',
    '', '', '', ''  # Some empty middle names
]

LAST_NAMES = [
    'Asare', 'Serwaa', 'Asante', 'Mensah', 'Osei', 'Owusu',
    'Agyei', 'Boateng', 'Agyemang', 'Appiah', 'Amoah', 'Adjei',
    'Frimpong', 'Acheampong', 'Gyasi', 'Opoku', 'Yeboah', 'Antwi',
    'Ofori', 'Anim', 'Addo', 'Agyapong', 'Amponsah', 'Afriyie',
    'Addai', 'Akoto', 'Ampofo', 'Ansah', 'Asiedu', 'Baffoe',
    'Bonsu', 'Danso', 'Darko', 'Donkor', 'Duah', 'Kuffour',
    'Kyei', 'Marfo', 'Ntim', 'Ofosu', 'Oppong', 'Poku',
    'Safo', 'Sarpong', 'Tawiah', 'Wiredu', 'Yiadom', 'Fordjour'
]

FAMILY_NAMES = [
    'Asare Family', 'Owusu Family', 'Mensah Family', 'Osei Family',
    'Boateng Family', 'Agyei Family', 'Appiah Family', 'Frimpong Family',
    'Antwi Family', 'Opoku Family', 'Yeboah Family', 'Ofori Family',
    'Amoah Family', 'Adjei Family', 'Agyemang Family', 'Gyasi Family',
    'Acheampong Family', 'Addai Family', 'Akoto Family', 'Amponsah Family',
    'Afriyie Family', 'Ampofo Family', 'Ansah Family', 'Asiedu Family',
    'Baffoe Family', 'Bonsu Family', 'Danso Family', 'Darko Family',
    'Donkor Family', 'Duah Family', 'Kuffour Family', 'Kyei Family'
]

CITIES = [
    'Accra', 'Kumasi', 'Takoradi', 'Tamale', 'Cape Coast',
    'Sunyani', 'Koforidua', 'Ho', 'Wa', 'Bolgatanga',
    'Tema', 'Obuasi', 'Tarkwa', 'Techiman', 'Nkawkaw',
    'Adum', 'Asokwa', 'Bantama', 'Atonsu', 'Ejisu'
]

STREETS = [
    'Independence Avenue', 'Kwame Nkrumah Street', 'Liberation Road',
    'Ring Road', 'Main Street', 'Church Street', 'Market Street',
    'Station Road', 'Hospital Road', 'School Lane', 'Palm Avenue',
    'Cedar Street', 'Oak Avenue', 'Mango Street', 'Cocoa Road'
]

OCCUPATIONS = [
    'Teacher', 'Nurse', 'Farmer', 'Trader', 'Driver', 'Mechanic',
    'Tailor', 'Hairdresser', 'Carpenter', 'Mason', 'Electrician',
    'Accountant', 'Engineer', 'Doctor', 'Lawyer', 'Banker',
    'Civil Servant', 'Business Owner', 'Student', 'Retired',
    'Seamstress', 'Shopkeeper', 'Caterer', 'Chef', 'Security Guard'
]

MEMBERSHIP_TYPES = ['communicant', 'baptized', 'confirmed', 'adherent', 'catechumen']
MEMBERSHIP_STATUS = ['active', 'active', 'active', 'active', 'inactive']  # More active members
MARITAL_STATUS = ['single', 'married', 'married', 'married', 'divorced', 'widowed']

def generate_phone():
    """Generate Ghanaian phone number"""
    prefixes = ['233', '0']
    prefix = random.choice(prefixes)
    if prefix == '233':
        # +233 24/54/55/59/20/50/23/27/28/26
        network = random.choice(['24', '54', '55', '59', '20', '50', '23', '27', '28', '26'])
        number = f"+{prefix}{network}{random.randint(1000000, 9999999)}"
    else:
        # 024/054/055/059/020/050/023/027/028/026
        network = random.choice(['24', '54', '55', '59', '20', '50', '23', '27', '28', '26'])
        number = f"0{network}{random.randint(1000000, 9999999)}"
    return number

def generate_email(first_name, last_name):
    """Generate email address"""
    domains = ['gmail.com', 'yahoo.com', 'hotmail.com', 'outlook.com']
    separators = ['.', '_', '']
    separator = random.choice(separators)
    
    # Some people don't have emails
    if random.random() < 0.3:  # 30% without email
        return ''
    
    email = f"{first_name.lower()}{separator}{last_name.lower()}@{random.choice(domains)}"
    return email

def generate_date_of_birth():
    """Generate random date of birth (ages 5-90)"""
    today = datetime.now()
    years_ago = random.randint(5, 90)
    days_range = years_ago * 365
    random_days = random.randint(0, days_range)
    birth_date = today - timedelta(days=random_days)
    return birth_date.strftime('%Y-%m-%d')

def generate_address(city):
    """Generate Ghanaian address"""
    house_number = random.randint(1, 999)
    street = random.choice(STREETS)
    return f"{house_number} {street}, {city}"

def generate_members(count=1000):
    """Generate member data"""
    members = []
    
    for i in range(count):
        gender = random.choice(['male', 'female'])
        
        if gender == 'male':
            first_name = random.choice(MALE_FIRST_NAMES)
        else:
            first_name = random.choice(FEMALE_FIRST_NAMES)
        
        middle_name = random.choice(MIDDLE_NAMES)
        last_name = random.choice(LAST_NAMES)
        email = generate_email(first_name, last_name)
        phone = generate_phone()
        date_of_birth = generate_date_of_birth()
        membership_type = random.choice(MEMBERSHIP_TYPES)
        membership_status = random.choice(MEMBERSHIP_STATUS)
        marital_status = random.choice(MARITAL_STATUS)
        city = random.choice(CITIES)
        address = generate_address(city)
        occupation = random.choice(OCCUPATIONS)
        
        # Generate emergency contact (usually a family member)
        emergency_gender = random.choice(['male', 'female'])
        if emergency_gender == 'male':
            emergency_first = random.choice(MALE_FIRST_NAMES)
        else:
            emergency_first = random.choice(FEMALE_FIRST_NAMES)
        
        emergency_contact_name = f"{emergency_first} {last_name}"
        emergency_contact_phone = generate_phone()
        family_name = random.choice(FAMILY_NAMES)
        
        member = {
            'first_name': first_name,
            'middle_name': middle_name,
            'last_name': last_name,
            'email': email,
            'phone': phone,
            'date_of_birth': date_of_birth,
            'gender': gender,
            'membership_type': membership_type,
            'membership_status': membership_status,
            'marital_status': marital_status,
            'address': address,
            'occupation': occupation,
            'emergency_contact_name': emergency_contact_name,
            'emergency_contact_phone': emergency_contact_phone,
            'family_name': family_name
        }
        
        members.append(member)
    
    return members

def create_excel_file(members, filename='member_bulk_import_1000.xlsx'):
    """Create Excel file with member data"""
    wb = Workbook()
    ws = wb.active
    ws.title = "Members"
    
    # Define headers
    headers = [
        'first_name', 'middle_name', 'last_name', 'email', 'phone',
        'date_of_birth', 'gender', 'membership_type', 'membership_status',
        'marital_status', 'address', 'occupation', 'emergency_contact_name',
        'emergency_contact_phone', 'family_name'
    ]
    
    # Write headers with styling
    header_fill = PatternFill(start_color='2563eb', end_color='2563eb', fill_type='solid')
    header_font = Font(bold=True, color='FFFFFF')
    
    for col_num, header in enumerate(headers, 1):
        cell = ws.cell(row=1, column=col_num)
        cell.value = header
        cell.fill = header_fill
        cell.font = header_font
        cell.alignment = Alignment(horizontal='center', vertical='center')
    
    # Write member data
    for row_num, member in enumerate(members, 2):
        ws.cell(row=row_num, column=1, value=member['first_name'])
        ws.cell(row=row_num, column=2, value=member['middle_name'])
        ws.cell(row=row_num, column=3, value=member['last_name'])
        ws.cell(row=row_num, column=4, value=member['email'])
        ws.cell(row=row_num, column=5, value=member['phone'])
        ws.cell(row=row_num, column=6, value=member['date_of_birth'])
        ws.cell(row=row_num, column=7, value=member['gender'])
        ws.cell(row=row_num, column=8, value=member['membership_type'])
        ws.cell(row=row_num, column=9, value=member['membership_status'])
        ws.cell(row=row_num, column=10, value=member['marital_status'])
        ws.cell(row=row_num, column=11, value=member['address'])
        ws.cell(row=row_num, column=12, value=member['occupation'])
        ws.cell(row=row_num, column=13, value=member['emergency_contact_name'])
        ws.cell(row=row_num, column=14, value=member['emergency_contact_phone'])
        ws.cell(row=row_num, column=15, value=member['family_name'])
    
    # Set column widths
    column_widths = {
        'A': 15, 'B': 15, 'C': 15, 'D': 30, 'E': 18,
        'F': 15, 'G': 12, 'H': 18, 'I': 18, 'J': 15,
        'K': 35, 'L': 20, 'M': 25, 'N': 18, 'O': 20
    }
    
    for col, width in column_widths.items():
        ws.column_dimensions[col].width = width
    
    # Save the workbook
    wb.save(filename)
    print(f"✅ Successfully created {filename} with {len(members)} members!")
    print(f"📍 File location: {filename}")

if __name__ == "__main__":
    print("🚀 Generating 1000 Ghanaian members for bulk import...")
    members = generate_members(1000)
    create_excel_file(members)
    print("✨ Done! You can now use this file for bulk member import.")
