<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class YearGroup extends Model
{
    protected $fillable = [
        'name',
        'abbreviation',
        'min_age',
        'max_age',
        'description',
        'color',
        'is_active',
        'sort_order'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'min_age' => 'integer',
        'max_age' => 'integer',
        'sort_order' => 'integer'
    ];

    /**
     * Get all members in this year group
     */
    public function members(): HasMany
    {
        return $this->hasMany(Member::class);
    }

    /**
     * Get active members count for this year group
     */
    public function getActiveMembersCountAttribute(): int
    {
        return $this->members()->where('membership_status', 'active')->count();
    }

    /**
     * Check if an age falls within this year group's range
     */
    public function includesAge(int $age): bool
    {
        return $age >= $this->min_age && $age <= $this->max_age;
    }

    /**
     * Get the appropriate year group for a given age
     */
    public static function getForAge(int $age): ?YearGroup
    {
        return static::where('is_active', true)
            ->where('min_age', '<=', $age)
            ->where('max_age', '>=', $age)
            ->orderBy('sort_order')
            ->first();
    }

    /**
     * Scope to get active year groups
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to order by sort order
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('min_age');
    }
}
