<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class InventoryItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'item_code',
        'name',
        'description',
        'category_id',
        'unit_of_measure',
        'unit_cost',
        'current_stock',
        'minimum_stock',
        'maximum_stock',
        'location',
        'supplier',
        'supplier_contact',
        'expiry_date',
        'barcode',
        'images',
        'notes',
        'is_active',
        'track_expiry'
    ];

    protected $casts = [
        'unit_cost' => 'decimal:2',
        'expiry_date' => 'date',
        'images' => 'array',
        'is_active' => 'boolean',
        'track_expiry' => 'boolean'
    ];

    // Relationships
    public function category()
    {
        return $this->belongsTo(InventoryCategory::class, 'category_id');
    }

    public function transactions()
    {
        return $this->hasMany(InventoryTransaction::class, 'item_id');
    }

    public function recentTransactions()
    {
        return $this->hasMany(InventoryTransaction::class, 'item_id')->orderBy('created_at', 'desc')->limit(10);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeLowStock($query)
    {
        return $query->whereRaw('current_stock <= minimum_stock');
    }

    public function scopeOutOfStock($query)
    {
        return $query->where('current_stock', 0);
    }

    public function scopeExpiringSoon($query, $days = 30)
    {
        return $query->where('track_expiry', true)
                    ->whereNotNull('expiry_date')
                    ->whereBetween('expiry_date', [now(), now()->addDays($days)]);
    }

    // Accessors
    public function getStockStatusAttribute()
    {
        if ($this->current_stock <= 0) {
            return 'out_of_stock';
        } elseif ($this->current_stock <= $this->minimum_stock) {
            return 'low_stock';
        } elseif ($this->maximum_stock && $this->current_stock >= $this->maximum_stock) {
            return 'overstock';
        }
        return 'in_stock';
    }

    public function getStockValueAttribute()
    {
        return $this->current_stock * $this->unit_cost;
    }

    public function getDaysToExpiryAttribute()
    {
        if (!$this->track_expiry || !$this->expiry_date) {
            return null;
        }
        return now()->diffInDays($this->expiry_date, false);
    }

    // Methods
    public function generateItemCode()
    {
        $prefix = $this->category ? strtoupper(substr($this->category->name, 0, 3)) : 'ITM';
        $lastItem = static::where('item_code', 'like', $prefix . '%')
                          ->orderBy('item_code', 'desc')
                          ->first();

        if ($lastItem) {
            $lastNumber = (int) substr($lastItem->item_code, strlen($prefix));
            $nextNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $nextNumber = '0001';
        }

        return $prefix . $nextNumber;
    }

    public function updateStock($quantity, $type = 'adjustment', $reason = null, $performedBy = null)
    {
        $previousStock = $this->current_stock;
        
        switch ($type) {
            case 'in':
                $this->current_stock += $quantity;
                break;
            case 'out':
                $this->current_stock -= $quantity;
                break;
            case 'adjustment':
                $this->current_stock = $quantity;
                break;
        }

        $this->save();

        // Create transaction record
        InventoryTransaction::create([
            'transaction_code' => $this->generateTransactionCode(),
            'item_id' => $this->id,
            'type' => $type,
            'quantity' => $type === 'adjustment' ? ($quantity - $previousStock) : $quantity,
            'previous_stock' => $previousStock,
            'new_stock' => $this->current_stock,
            'reason' => $reason,
            'performed_by' => $performedBy,
            'transaction_date' => now()->toDateString()
        ]);
    }

    private function generateTransactionCode()
    {
        $prefix = 'TXN';
        $date = now()->format('Ymd');
        $lastTransaction = InventoryTransaction::where('transaction_code', 'like', $prefix . $date . '%')
                                             ->orderBy('transaction_code', 'desc')
                                             ->first();

        if ($lastTransaction) {
            $lastNumber = (int) substr($lastTransaction->transaction_code, -4);
            $nextNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $nextNumber = '0001';
        }

        return $prefix . $date . $nextNumber;
    }

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($item) {
            if (empty($item->item_code)) {
                $item->item_code = $item->generateItemCode();
            }
        });
    }
}
