<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class BirthdayMessage extends Model
{
    protected $fillable = [
        'member_id',
        'message_type',
        'message_content',
        'phone_number',
        'email_address',
        'status',
        'sent_at',
        'birthday_date',
        'error_message',
        'delivery_details'
    ];

    protected $casts = [
        'sent_at' => 'datetime',
        'birthday_date' => 'date',
        'delivery_details' => 'array'
    ];

    /**
     * Get the member that owns the birthday message
     */
    public function member(): BelongsTo
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Scope for pending messages
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for sent messages
     */
    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    /**
     * Scope for failed messages
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope for today's birthday messages
     */
    public function scopeToday($query)
    {
        return $query->whereDate('birthday_date', Carbon::today());
    }

    /**
     * Scope for this week's birthday messages
     */
    public function scopeThisWeek($query)
    {
        return $query->whereBetween('birthday_date', [
            Carbon::now()->startOfWeek(),
            Carbon::now()->endOfWeek()
        ]);
    }

    /**
     * Mark message as sent
     */
    public function markAsSent($deliveryDetails = null): void
    {
        $this->update([
            'status' => 'sent',
            'sent_at' => now(),
            'delivery_details' => $deliveryDetails
        ]);
    }

    /**
     * Mark message as failed
     */
    public function markAsFailed($errorMessage): void
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage
        ]);
    }
}
