<?php

namespace App\Http\Controllers;

use App\Models\YearGroup;
use App\Models\Member;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class YearGroupMemberController extends Controller
{
    /**
     * Display year groups with member counts
     */
    public function index()
    {
        $yearGroups = YearGroup::active()
            ->ordered()
            ->withCount(['members' => function($query) {
                $query->where('membership_status', 'active');
            }])
            ->get();
        
        $totalMembers = Member::where('membership_status', 'active')
            ->whereNotNull('year_group_id')
            ->count();
        
        $unassignedMembers = Member::where('membership_status', 'active')
            ->whereNull('year_group_id')
            ->count();
        
        return view('year-groups.members.index', compact('yearGroups', 'totalMembers', 'unassignedMembers'));
    }
    
    /**
     * Show members for a specific year group
     */
    public function show(YearGroup $yearGroup, Request $request)
    {
        $query = $yearGroup->members()
            ->where('membership_status', 'active')
            ->orderBy('first_name')
            ->orderBy('last_name');
        
        // Apply filters
        if ($request->filled('gender')) {
            $query->where('gender', $request->gender);
        }
        
        if ($request->filled('membership_type')) {
            $query->where('membership_type', $request->membership_type);
        }
        
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('middle_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('member_id', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }
        
        $members = $query->paginate(50);
        
        // Get all year groups for the dropdown
        $yearGroups = YearGroup::active()->ordered()->get();
        
        return view('year-groups.members.show', compact('yearGroup', 'members', 'yearGroups'));
    }
    
    /**
     * Print member list for a year group
     */
    public function print(YearGroup $yearGroup, Request $request)
    {
        $query = $yearGroup->members()
            ->where('membership_status', 'active')
            ->orderBy('first_name')
            ->orderBy('last_name');
        
        // Apply same filters as show
        if ($request->filled('gender')) {
            $query->where('gender', $request->gender);
        }
        
        if ($request->filled('membership_type')) {
            $query->where('membership_type', $request->membership_type);
        }
        
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('middle_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('member_id', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }
        
        $members = $query->get();
        
        $pdf = PDF::loadView('year-groups.members.print', compact('yearGroup', 'members'))
            ->setPaper('a4', 'portrait');
        
        return $pdf->download($yearGroup->name . '-members-' . date('Y-m-d') . '.pdf');
    }
    
    /**
     * Export members to Excel
     */
    public function export(YearGroup $yearGroup, Request $request)
    {
        // Future implementation for Excel export
        return redirect()->back()->with('info', 'Excel export coming soon!');
    }
}
