<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PermissionController extends Controller
{
    /**
     * Display a listing of permissions.
     */
    public function index()
    {
        $permissions = Permission::withCount('roles')
                                ->orderBy('module')
                                ->orderBy('name')
                                ->get()
                                ->groupBy('module');

        $stats = [
            'total_permissions' => Permission::count(),
            'total_modules' => Permission::distinct('module')->count('module'),
            'assigned_permissions' => Permission::has('roles')->count(),
        ];

        return view('permissions.index', compact('permissions', 'stats'));
    }

    /**
     * Show the form for creating a new permission.
     */
    public function create()
    {
        $modules = Permission::distinct()->pluck('module')->toArray();
        return view('permissions.create', compact('modules'));
    }

    /**
     * Store a newly created permission in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name',
            'slug' => 'nullable|string|max:255|unique:permissions,slug',
            'module' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        Permission::create([
            'name' => $validated['name'],
            'slug' => $validated['slug'] ?? Str::slug($validated['name']),
            'module' => $validated['module'],
            'description' => $validated['description'] ?? null,
        ]);

        return redirect()->route('permissions.index')
                        ->with('success', 'Permission created successfully!');
    }

    /**
     * Display the specified permission.
     */
    public function show(Permission $permission)
    {
        $permission->load('roles.users');
        return view('permissions.show', compact('permission'));
    }

    /**
     * Show the form for editing the specified permission.
     */
    public function edit(Permission $permission)
    {
        $modules = Permission::distinct()->pluck('module')->toArray();
        return view('permissions.edit', compact('permission', 'modules'));
    }

    /**
     * Update the specified permission in storage.
     */
    public function update(Request $request, Permission $permission)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name,' . $permission->id,
            'slug' => 'nullable|string|max:255|unique:permissions,slug,' . $permission->id,
            'module' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $permission->update([
            'name' => $validated['name'],
            'slug' => $validated['slug'] ?? Str::slug($validated['name']),
            'module' => $validated['module'],
            'description' => $validated['description'] ?? null,
        ]);

        return redirect()->route('permissions.index')
                        ->with('success', 'Permission updated successfully!');
    }

    /**
     * Remove the specified permission from storage.
     */
    public function destroy(Permission $permission)
    {
        // Detach from all roles
        $permission->roles()->detach();
        $permission->delete();

        return redirect()->route('permissions.index')
                        ->with('success', 'Permission deleted successfully!');
    }
}
