<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Family;
use App\Mail\MemberRegistrationEmail;
use App\Mail\MemberApprovedEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class MemberRegistrationController extends Controller
{
    /**
     * Show member registration form
     */
    public function showForm()
    {
        return view('member-registration.create');
    }

    /**
     * Register a new member
     */
    public function register(Request $request)
    {
        $validated = $request->validate([
            // Personal Information
            'title' => 'nullable|string|max:10',
            'first_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'last_name' => 'required|string|max:100',
            'date_of_birth' => 'required|date|before:today',
            'gender' => 'required|in:male,female',
            'marital_status' => 'required|in:single,married,divorced,widowed',
            'church_affiliation' => 'nullable|string|max:255',
            
            // Contact Information
            'phone' => 'required|string|max:20|unique:members,phone',
            'whatsapp_phone' => 'nullable|string|max:20',
            'alternate_phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:members,email',
            'address' => 'required|string|max:500',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'required|string|max:100',
            
            // Professional Information
            'occupation' => 'nullable|string|max:255',
            'employer' => 'nullable|string|max:255',
            
            // Membership Information
            'membership_type' => 'required|in:communicant,baptized,confirmed,long_distance,adherent,catechumen',
            'membership_date' => 'required|date',
            'chapter' => 'nullable|string|max:100',
            
            // Emergency Contact
            'emergency_contact_name' => 'required|string|max:255',
            'emergency_contact_phone' => 'required|string|max:20',
            'emergency_contact_relationship' => 'required|string|max:100',
            
            // Guardian Information (for minors)
            'guardian_name' => 'nullable|string|max:255',
            'guardian_relationship' => 'nullable|string|max:100',
            'guardian_phone' => 'nullable|string|max:20',
            'guardian_address' => 'nullable|string|max:500',
            
            // Health Information
            'medical_conditions' => 'nullable|string|max:1000',
            'special_needs' => 'nullable|string|max:1000',
            'physically_challenged' => 'boolean',
            
            // Baptism Details
            'is_baptized' => 'boolean',
            'baptism_date' => 'nullable|date|required_if:is_baptized,true',
            'place_of_baptism' => 'nullable|string|max:255|required_if:is_baptized,true',
            'baptism_type' => 'nullable|in:sprinkling,immersion,pouring,water,other|required_if:is_baptized,true',
            'baptism_officiating_minister' => 'nullable|string|max:255',
            
            // Confirmation Details
            'is_confirmed' => 'boolean',
            'confirmation_date' => 'nullable|date|required_if:is_confirmed,true',
            'place_of_confirmation' => 'nullable|string|max:255|required_if:is_confirmed,true',
            'age_confirmed' => 'nullable|integer|min:1|max:100',
            'confirmation_bible_verse' => 'nullable|string|max:500',
            'confirmation_officiating_minister' => 'nullable|string|max:255',
            
            // Marriage Details
            'is_married' => 'boolean',
            'marriage_date' => 'nullable|date|required_if:is_married,true',
            'place_of_marriage' => 'nullable|string|max:255|required_if:is_married,true',
            'marriage_type' => 'nullable|in:traditional,customary,ordinance,other|required_if:is_married,true',
            'marriage_officiating_minister' => 'nullable|string|max:255',
            
            // Family Information
            'has_family' => 'boolean',
            'family_id' => 'nullable|exists:families,id',
            'relationship_to_head' => 'nullable|string|max:100',
            
            // Additional Information
            'skills_talents' => 'nullable|array',
            'interests' => 'nullable|array',
            'notes' => 'nullable|string|max:2000',
            
            // Communication Preferences
            'receive_newsletter' => 'boolean',
            'receive_sms' => 'boolean',
            
            // Photo
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            
            // Password (optional for self-registration)
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        // Handle photo upload
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('member-photos', 'public');
            $validated['photo_path'] = $photoPath;
        }

        // Hash password if provided
        if (!empty($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        }

        // Set default values
        $validated['is_active'] = true;
        $validated['membership_status'] = 'pending'; // Requires approval

        // Create the member
        $member = Member::create($validated);

        // Handle family association
        if ($validated['has_family'] && !empty($validated['family_id'])) {
            $member->family_id = $validated['family_id'];
            $member->save();
        } elseif ($validated['has_family'] && empty($validated['family_id'])) {
            // Create new family if member indicates they have family but no family_id provided
            $family = Family::create([
                'family_name' => $member->last_name . ' Family',
                'head_of_family_id' => $member->id,
                'created_by' => auth()->id() ?? 1
            ]);
            
            $member->family_id = $family->id;
            $member->relationship_to_head = 'head';
            $member->save();
        }

        // Send confirmation email to the member
        if ($member->email) {
            try {
                Mail::to($member->email)->send(new MemberRegistrationEmail($member));
            } catch (\Exception $e) {
                // Log the error but don't stop the registration process
                \Log::error('Failed to send registration email: ' . $e->getMessage());
            }
        }

        return redirect()->route('member.register.success')
                        ->with('success', 'Registration submitted successfully! Your membership is pending approval.');
    }

    /**
     * Show success page
     */
    public function success()
    {
        return view('member-registration.success');
    }

    /**
     * Show pending members for admin approval
     */
    public function pendingMembers()
    {
        $pendingMembers = Member::where('membership_status', 'pending')
                                ->orderBy('created_at', 'desc')
                                ->get();
        
        return view('members.pending', compact('pendingMembers'));
    }

    /**
     * Approve a pending member
     */
    public function approve(Member $member)
    {
        if ($member->membership_status !== 'pending') {
            return redirect()->back()->with('error', 'This member is not pending approval.');
        }

        // Generate a random password for the member
        $generatedPassword = Str::random(10);
        
        $member->membership_status = 'active';
        $member->password = Hash::make($generatedPassword);
        $member->save();

        // Send approval email to the member with login credentials
        if ($member->email) {
            try {
                Mail::to($member->email)->send(new MemberApprovedEmail($member, $generatedPassword));
            } catch (\Exception $e) {
                \Log::error('Failed to send approval email: ' . $e->getMessage());
            }
        }

        return redirect()->back()->with('success', "Member {$member->full_name} has been approved and notified via email with login credentials!");
    }

    /**
     * Reject a pending member
     */
    public function reject(Member $member)
    {
        if ($member->membership_status !== 'pending') {
            return redirect()->back()->with('error', 'This member is not pending approval.');
        }

        $member->membership_status = 'rejected';
        $member->is_active = false;
        $member->save();

        return redirect()->back()->with('success', "Member {$member->full_name} has been rejected.");
    }
}
