<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\InventoryItem;
use App\Models\InventoryCategory;
use App\Models\Setting;

class InventoryItemController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get organization settings for dynamic content
        $organizationSettings = [
            'name' => Setting::getValue('organization_name', 'general', 'Presbyterian Church of Ghana'),
            'slogan' => Setting::getValue('organization_slogan', 'general', 'Church Management System'),
            'logo' => Setting::getValue('organization_logo', 'general')
        ];

        $query = InventoryItem::with(['category', 'recentTransactions']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('item_code', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('location', 'like', "%{$search}%");
            });
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Filter by stock status
        if ($request->filled('stock_status')) {
            switch ($request->stock_status) {
                case 'low_stock':
                    $query->lowStock();
                    break;
                case 'out_of_stock':
                    $query->outOfStock();
                    break;
                case 'in_stock':
                    $query->where('current_stock', '>', 0)
                          ->whereRaw('current_stock > minimum_stock');
                    break;
            }
        }

        // Filter by active status
        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        $items = $query->orderBy('name')->paginate(20);
        $categories = InventoryCategory::where('is_active', true)->orderBy('name')->get();

        return view('inventory.items.index', compact('items', 'categories', 'organizationSettings'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = InventoryCategory::where('is_active', true)->orderBy('name')->get();
        return view('inventory.items.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:inventory_categories,id',
            'unit_of_measure' => 'required|string|max:50',
            'unit_cost' => 'required|numeric|min:0',
            'current_stock' => 'required|integer|min:0',
            'minimum_stock' => 'required|integer|min:0',
            'maximum_stock' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'supplier' => 'nullable|string|max:255',
            'supplier_contact' => 'nullable|string|max:255',
            'expiry_date' => 'nullable|date|after:today',
            'barcode' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'track_expiry' => 'boolean'
        ]);

        $item = InventoryItem::create($validated);

        // Create initial stock transaction if stock > 0
        if ($item->current_stock > 0) {
            $item->updateStock($item->current_stock, 'in', 'Initial stock entry', auth()->user()->name ?? 'System');
        }

        return redirect()->route('inventory.items.index')
                        ->with('success', 'Item created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(InventoryItem $item)
    {
        $item->load(['category', 'transactions' => function($query) {
            $query->with('member')->orderBy('created_at', 'desc');
        }]);

        return view('inventory.items.show', compact('item'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(InventoryItem $item)
    {
        $categories = InventoryCategory::where('is_active', true)->orderBy('name')->get();
        return view('inventory.items.edit', compact('item', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, InventoryItem $item)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:inventory_categories,id',
            'unit_of_measure' => 'required|string|max:50',
            'unit_cost' => 'required|numeric|min:0',
            'minimum_stock' => 'required|integer|min:0',
            'maximum_stock' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'supplier' => 'nullable|string|max:255',
            'supplier_contact' => 'nullable|string|max:255',
            'expiry_date' => 'nullable|date|after:today',
            'barcode' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'track_expiry' => 'boolean',
            'is_active' => 'boolean'
        ]);

        $item->update($validated);

        return redirect()->route('inventory.items.index')
                        ->with('success', 'Item updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(InventoryItem $item)
    {
        $item->delete();

        return redirect()->route('inventory.items.index')
                        ->with('success', 'Item deleted successfully!');
    }

    /**
     * Update item stock
     */
    public function updateStock(Request $request, InventoryItem $item)
    {
        $validated = $request->validate([
            'type' => 'required|in:in,out,adjustment',
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string|max:255',
            'reference_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string'
        ]);

        $item->updateStock(
            $validated['quantity'],
            $validated['type'],
            $validated['reason'],
            auth()->user()->name ?? 'System'
        );

        return redirect()->back()->with('success', 'Stock updated successfully!');
    }
}
