<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\InventoryItem;
use App\Models\InventoryCategory;
use App\Models\InventoryTransaction;
use App\Models\Setting;

class InventoryController extends Controller
{
    /**
     * Display the inventory dashboard
     */
    public function index()
    {
        // Get organization settings for dynamic content
        $organizationSettings = [
            'name' => Setting::getValue('organization_name', 'general', 'Presbyterian Church of Ghana'),
            'slogan' => Setting::getValue('organization_slogan', 'general', 'Church Management System'),
            'logo' => Setting::getValue('organization_logo', 'general')
        ];

        // Calculate statistics
        $stats = [
            'total_items' => InventoryItem::count(),
            'active_items' => InventoryItem::where('is_active', true)->count(),
            'total_categories' => InventoryCategory::where('is_active', true)->count(),
            'low_stock_items' => InventoryItem::lowStock()->count(),
            'out_of_stock_items' => InventoryItem::outOfStock()->count(),
            'expiring_soon' => InventoryItem::expiringSoon(30)->count(),
            'total_stock_value' => InventoryItem::where('is_active', true)->sum(\DB::raw('current_stock * unit_cost')),
            'recent_transactions' => InventoryTransaction::recent(7)->count()
        ];

        // Get recent transactions
        $recentTransactions = InventoryTransaction::with(['item', 'member'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Get low stock items
        $lowStockItems = InventoryItem::with('category')
            ->lowStock()
            ->where('is_active', true)
            ->orderBy('current_stock', 'asc')
            ->limit(10)
            ->get();

        // Get expiring items
        $expiringItems = InventoryItem::with('category')
            ->expiringSoon(30)
            ->where('is_active', true)
            ->orderBy('expiry_date', 'asc')
            ->limit(10)
            ->get();

        // Get categories with item counts
        $categories = InventoryCategory::withCount(['activeItems'])
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.index', compact(
            'organizationSettings',
            'stats',
            'recentTransactions',
            'lowStockItems',
            'expiringItems',
            'categories'
        ));
    }

    /**
     * Show stock alerts
     */
    public function alerts()
    {
        $lowStockItems = InventoryItem::with('category')
            ->lowStock()
            ->where('is_active', true)
            ->orderBy('current_stock', 'asc')
            ->paginate(20);

        $outOfStockItems = InventoryItem::with('category')
            ->outOfStock()
            ->where('is_active', true)
            ->orderBy('name')
            ->paginate(20);

        $expiringItems = InventoryItem::with('category')
            ->expiringSoon(30)
            ->where('is_active', true)
            ->orderBy('expiry_date', 'asc')
            ->paginate(20);

        return view('inventory.alerts', compact(
            'lowStockItems',
            'outOfStockItems',
            'expiringItems'
        ));
    }

    /**
     * Show inventory reports
     */
    public function reports(Request $request)
    {
        // Get organization settings for dynamic content
        $organizationSettings = [
            'name' => Setting::getValue('organization_name', 'general', 'Presbyterian Church of Ghana'),
            'slogan' => Setting::getValue('organization_slogan', 'general', 'Church Management System'),
            'logo' => Setting::getValue('organization_logo', 'general')
        ];

        $dateRange = $request->get('range', '30'); // Default 30 days
        $startDate = now()->subDays($dateRange);
        $endDate = now();

        // Calculate statistics
        $stats = [
            'total_items' => InventoryItem::count(),
            'total_categories' => InventoryCategory::where('is_active', true)->count(),
            'low_stock_items' => InventoryItem::lowStock()->count(),
            'total_stock_value' => InventoryItem::where('is_active', true)->sum(\DB::raw('current_stock * unit_cost'))
        ];

        // Get categories for filter dropdown
        $categories = InventoryCategory::where('is_active', true)
            ->orderBy('name')
            ->get();

        // Stock movement report
        $stockMovements = InventoryTransaction::with(['item.category'])
            ->byDateRange($startDate, $endDate)
            ->orderBy('transaction_date', 'desc')
            ->paginate(50);

        // Category-wise stock value
        $categoryStats = InventoryCategory::with('items')
            ->where('is_active', true)
            ->get()
            ->map(function ($category) {
                return [
                    'name' => $category->name,
                    'items_count' => $category->activeItems()->count(),
                    'total_stock' => $category->activeItems()->sum('current_stock'),
                    'total_value' => $category->activeItems()->sum(\DB::raw('current_stock * unit_cost'))
                ];
            });

        return view('inventory.reports', compact(
            'organizationSettings',
            'stats',
            'categories',
            'stockMovements',
            'categoryStats',
            'dateRange',
            'startDate',
            'endDate'
        ));
    }
}
