<?php

namespace App\Console\Commands;

use App\Models\Member;
use App\Models\YearGroup;
use Illuminate\Console\Command;
use Carbon\Carbon;

class AssignYearGroups extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'members:assign-year-groups {--force : Force reassignment for all members}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Auto-assign year groups to members based on their age';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $force = $this->option('force');
        
        $this->info('🔄 Starting year group assignment...');
        
        // Get members to process
        if ($force) {
            $members = Member::whereNotNull('date_of_birth')->get();
            $this->info('Processing all members with date of birth...');
        } else {
            $members = Member::whereNull('year_group_id')
                ->whereNotNull('date_of_birth')
                ->get();
            $this->info('Processing members without year group assignment...');
        }
        
        if ($members->isEmpty()) {
            $this->warn('⚠️  No members found to process.');
            return Command::SUCCESS;
        }
        
        $this->info("Found {$members->count()} member(s) to process.");
        
        $progressBar = $this->output->createProgressBar($members->count());
        $progressBar->start();
        
        $assigned = 0;
        $skipped = 0;
        $noGroup = 0;
        
        foreach ($members as $member) {
            try {
                $dateOfBirth = Carbon::parse($member->date_of_birth);
                $age = $dateOfBirth->age;
                
                $yearGroup = YearGroup::getForAge($age);
                
                if ($yearGroup) {
                    $member->year_group_id = $yearGroup->id;
                    $member->save();
                    $assigned++;
                } else {
                    $noGroup++;
                }
            } catch (\Exception $e) {
                $skipped++;
            }
            
            $progressBar->advance();
        }
        
        $progressBar->finish();
        $this->newLine(2);
        
        // Display summary
        $this->info('✅ Year group assignment completed!');
        $this->table(
            ['Status', 'Count'],
            [
                ['Assigned', $assigned],
                ['No matching group', $noGroup],
                ['Skipped (errors)', $skipped],
            ]
        );
        
        if ($noGroup > 0) {
            $this->warn("⚠️  {$noGroup} member(s) have no matching year group. Consider creating year groups for these ages.");
        }
        
        return Command::SUCCESS;
    }
}
